/**
 * Mailchimp Integration Service
 * Sync leads to Mailchimp lists with tags
 */

const axios = require('axios');
const { pool } = require('../utils/db');
const { decrypt } = require('../utils/encryption');

class MailchimpService {
  constructor(apiKey) {
    this.apiKey = apiKey;
    this.datacenter = apiKey.split('-').pop();
    this.baseUrl = `https://${this.datacenter}.api.mailchimp.com/3.0`;
  }

  /**
   * Make authenticated request to Mailchimp API
   */
  async request(method, endpoint, data = null) {
    try {
      const response = await axios({
        method,
        url: `${this.baseUrl}${endpoint}`,
        headers: {
          'Authorization': `Bearer ${this.apiKey}`,
          'Content-Type': 'application/json'
        },
        data
      });
      return response.data;
    } catch (error) {
      console.error('Mailchimp API error:', error.response?.data || error.message);
      throw new Error(error.response?.data?.detail || 'Mailchimp API error');
    }
  }

  /**
   * Get all lists (audiences)
   */
  async getLists() {
    const data = await this.request('GET', '/lists?count=100');
    return data.lists.map(list => ({
      id: list.id,
      name: list.name,
      memberCount: list.stats.member_count
    }));
  }

  /**
   * Get list tags
   */
  async getTags(listId) {
    const data = await this.request('GET', `/lists/${listId}/segments?type=static&count=100`);
    return data.segments.map(seg => ({
      id: seg.id,
      name: seg.name,
      memberCount: seg.member_count
    }));
  }

  /**
   * Add/update subscriber to list
   */
  async addSubscriber(listId, subscriber, tags = []) {
    const { email, firstName, lastName, phone, customFields } = subscriber;
    
    // Hash email for Mailchimp subscriber endpoint
    const crypto = require('crypto');
    const emailHash = crypto.createHash('md5').update(email.toLowerCase()).digest('hex');

    const mergeFields = {
      FNAME: firstName || '',
      LNAME: lastName || '',
      PHONE: phone || '',
      ...customFields
    };

    try {
      // Add/update subscriber
      await this.request('PUT', `/lists/${listId}/members/${emailHash}`, {
        email_address: email.toLowerCase(),
        status_if_new: 'subscribed',
        merge_fields: mergeFields
      });

      // Add tags if any
      if (tags.length > 0) {
        await this.addTags(listId, email, tags);
      }

      return true;
    } catch (error) {
      console.error('Add subscriber error:', error);
      throw error;
    }
  }

  /**
   * Add tags to subscriber
   */
  async addTags(listId, email, tags) {
    const crypto = require('crypto');
    const emailHash = crypto.createHash('md5').update(email.toLowerCase()).digest('hex');

    const tagData = tags.map(tag => ({
      name: tag,
      status: 'active'
    }));

    await this.request('POST', `/lists/${listId}/members/${emailHash}/tags`, {
      tags: tagData
    });
  }

  /**
   * Create a tag if it doesn't exist
   */
  async createTag(listId, tagName) {
    try {
      const data = await this.request('POST', `/lists/${listId}/segments`, {
        name: tagName,
        static_segment: []
      });
      return data.id;
    } catch (error) {
      // Tag might already exist, that's ok
      if (error.message.includes('already exists')) {
        return null;
      }
      throw error;
    }
  }
}

/**
 * Sync lead to Mailchimp
 * Called after quiz completion
 */
async function syncLeadToMailchimp(leadId) {
  try {
    // Get lead with integration settings
    const [leads] = await pool.execute(
      `SELECT l.*, qi.tag_mapping, qi.field_mapping, i.credentials, i.settings
       FROM leads l
       JOIN quiz_integrations qi ON l.quiz_id = qi.quiz_id
       JOIN integrations i ON qi.integration_id = i.id
       WHERE l.id = ? AND i.platform = 'mailchimp' AND i.enabled = 1`,
      [leadId]
    );

    if (leads.length === 0) {
      return null; // No Mailchimp integration for this quiz
    }

    const lead = leads[0];
    const apiKey = decrypt(lead.credentials);
    const settings = JSON.parse(lead.settings || '{}');
    const tagMapping = JSON.parse(lead.tag_mapping || '{}');

    if (!apiKey || !settings.listId) {
      console.log('Mailchimp not configured properly');
      return null;
    }

    const mailchimp = new MailchimpService(apiKey);

    // Parse lead name
    const nameParts = (lead.name || '').split(' ');
    const firstName = nameParts[0] || '';
    const lastName = nameParts.slice(1).join(' ') || '';

    // Get tags from lead + any mapped tags
    let tags = [];
    if (lead.tags) {
      const leadTags = JSON.parse(lead.tags);
      tags = [...leadTags];
    }

    // Add mapped tags based on result
    if (lead.result_page_id && tagMapping[lead.result_page_id]) {
      tags.push(tagMapping[lead.result_page_id]);
    }

    // Sync to Mailchimp
    await mailchimp.addSubscriber(settings.listId, {
      email: lead.email,
      firstName,
      lastName,
      phone: lead.phone,
      customFields: JSON.parse(lead.custom_fields || '{}')
    }, tags);

    // Mark as synced
    await pool.execute(
      'UPDATE leads SET synced_mailchimp = 1 WHERE id = ?',
      [leadId]
    );

    // Update integration last sync time
    await pool.execute(
      'UPDATE integrations SET last_sync = NOW(), last_error = NULL WHERE id = ?',
      [lead.integration_id]
    );

    return true;
  } catch (error) {
    console.error('Mailchimp sync error:', error);
    
    // Log error to integration
    await pool.execute(
      'UPDATE integrations SET last_error = ? WHERE platform = "mailchimp"',
      [error.message]
    );
    
    throw error;
  }
}

module.exports = {
  MailchimpService,
  syncLeadToMailchimp
};
