/**
 * Google Sheets Integration Service
 * Sync leads to Google Sheets in real-time
 */

const { google } = require('googleapis');
const { pool } = require('../utils/db');

/**
 * Get authenticated Google Sheets client
 */
function getSheetsClient() {
  const auth = new google.auth.GoogleAuth({
    credentials: {
      client_email: process.env.GOOGLE_SERVICE_ACCOUNT_EMAIL,
      private_key: process.env.GOOGLE_PRIVATE_KEY?.replace(/\\n/g, '\n')
    },
    scopes: ['https://www.googleapis.com/auth/spreadsheets']
  });

  return google.sheets({ version: 'v4', auth });
}

class GoogleSheetsService {
  constructor(spreadsheetId, sheetName = 'Sheet1') {
    this.spreadsheetId = spreadsheetId;
    this.sheetName = sheetName;
    this.sheets = getSheetsClient();
  }

  /**
   * Get spreadsheet info
   */
  async getSpreadsheetInfo() {
    const response = await this.sheets.spreadsheets.get({
      spreadsheetId: this.spreadsheetId,
      fields: 'properties.title,sheets.properties.title'
    });

    return {
      title: response.data.properties.title,
      sheets: response.data.sheets.map(s => s.properties.title)
    };
  }

  /**
   * Get header row
   */
  async getHeaders() {
    const response = await this.sheets.spreadsheets.values.get({
      spreadsheetId: this.spreadsheetId,
      range: `${this.sheetName}!1:1`
    });

    return response.data.values?.[0] || [];
  }

  /**
   * Set header row
   */
  async setHeaders(headers) {
    await this.sheets.spreadsheets.values.update({
      spreadsheetId: this.spreadsheetId,
      range: `${this.sheetName}!A1`,
      valueInputOption: 'RAW',
      requestBody: {
        values: [headers]
      }
    });
  }

  /**
   * Append row to sheet
   */
  async appendRow(values) {
    await this.sheets.spreadsheets.values.append({
      spreadsheetId: this.spreadsheetId,
      range: `${this.sheetName}!A:Z`,
      valueInputOption: 'RAW',
      insertDataOption: 'INSERT_ROWS',
      requestBody: {
        values: [values]
      }
    });
  }

  /**
   * Append lead data with auto-header creation
   */
  async appendLead(leadData) {
    const defaultHeaders = [
      'Date',
      'Email',
      'Name',
      'Phone',
      'Quiz',
      'Result',
      'Score',
      'Tags',
      'Completed',
      'UTM Source',
      'UTM Medium',
      'UTM Campaign'
    ];

    // Check if headers exist
    let headers = await this.getHeaders();
    
    if (headers.length === 0) {
      // Add custom field headers if any
      const customFieldKeys = Object.keys(leadData.customFields || {});
      headers = [...defaultHeaders, ...customFieldKeys];
      await this.setHeaders(headers);
    }

    // Map lead data to row values
    const row = [
      new Date().toISOString(),
      leadData.email || '',
      leadData.name || '',
      leadData.phone || '',
      leadData.quizTitle || '',
      leadData.resultTitle || '',
      leadData.score || '',
      (leadData.tags || []).join(', '),
      leadData.completed ? 'Yes' : 'No',
      leadData.utmSource || '',
      leadData.utmMedium || '',
      leadData.utmCampaign || ''
    ];

    // Add custom field values
    const customFields = leadData.customFields || {};
    for (let i = defaultHeaders.length; i < headers.length; i++) {
      row.push(customFields[headers[i]] || '');
    }

    await this.appendRow(row);
  }
}

/**
 * Sync lead to Google Sheets
 * Called after quiz completion
 */
async function syncLeadToSheets(leadId) {
  try {
    // Check if Google Sheets is configured
    if (!process.env.GOOGLE_SERVICE_ACCOUNT_EMAIL || !process.env.GOOGLE_PRIVATE_KEY) {
      console.log('Google Sheets not configured');
      return null;
    }

    // Get lead with integration settings
    const [leads] = await pool.execute(
      `SELECT l.*, q.title as quiz_title, p.title as result_title,
              qi.field_mapping, i.settings, i.id as integration_id
       FROM leads l
       JOIN quizzes q ON l.quiz_id = q.id
       LEFT JOIN pages p ON l.result_page_id = p.id
       JOIN quiz_integrations qi ON l.quiz_id = qi.quiz_id
       JOIN integrations i ON qi.integration_id = i.id
       WHERE l.id = ? AND i.platform = 'sheets' AND i.enabled = 1`,
      [leadId]
    );

    if (leads.length === 0) {
      return null; // No Sheets integration for this quiz
    }

    const lead = leads[0];
    const settings = JSON.parse(lead.settings || '{}');

    if (!settings.spreadsheetId) {
      console.log('Sheets not configured properly');
      return null;
    }

    const sheets = new GoogleSheetsService(
      settings.spreadsheetId,
      settings.sheetName || 'Quiz Leads'
    );

    // Prepare lead data
    const leadData = {
      email: lead.email,
      name: lead.name,
      phone: lead.phone,
      quizTitle: lead.quiz_title,
      resultTitle: lead.result_title,
      score: lead.score,
      tags: lead.tags ? JSON.parse(lead.tags) : [],
      completed: lead.completed_quiz,
      utmSource: lead.utm_source,
      utmMedium: lead.utm_medium,
      utmCampaign: lead.utm_campaign,
      customFields: lead.custom_fields ? JSON.parse(lead.custom_fields) : {}
    };

    // Append to sheet
    await sheets.appendLead(leadData);

    // Mark as synced
    await pool.execute(
      'UPDATE leads SET synced_sheets = 1 WHERE id = ?',
      [leadId]
    );

    // Update integration last sync time
    await pool.execute(
      'UPDATE integrations SET last_sync = NOW(), last_error = NULL WHERE id = ?',
      [lead.integration_id]
    );

    return true;
  } catch (error) {
    console.error('Sheets sync error:', error);
    
    // Log error
    await pool.execute(
      `UPDATE integrations SET last_error = ? 
       WHERE platform = 'sheets' AND account_id = (SELECT account_id FROM leads WHERE id = ?)`,
      [error.message, leadId]
    );
    
    throw error;
  }
}

module.exports = {
  GoogleSheetsService,
  syncLeadToSheets
};
