/**
 * Upload Routes - Single Tenant Version
 * Image/media uploads - local or Bunny CDN
 */

const express = require('express');
const router = express.Router();
const multer = require('multer');
const path = require('path');
const fs = require('fs').promises;
const axios = require('axios');
const { v4: uuidv4 } = require('uuid');
const { authenticate } = require('../middleware/auth');
const { asyncHandler } = require('../middleware/errorHandler');

// Configure multer
const upload = multer({
  storage: multer.memoryStorage(),
  limits: { fileSize: 10 * 1024 * 1024 }, // 10MB
  fileFilter: (req, file, cb) => {
    const allowed = ['image/jpeg', 'image/png', 'image/gif', 'image/webp', 'image/svg+xml', 'video/mp4', 'video/webm'];
    cb(null, allowed.includes(file.mimetype));
  }
});

// All routes require authentication
router.use(authenticate);

/**
 * POST /api/upload/image
 * Upload single image
 */
router.post('/image', upload.single('image'), asyncHandler(async (req, res) => {
  if (!req.file) {
    return res.status(400).json({ error: 'No file uploaded' });
  }

  const file = req.file;
  const ext = path.extname(file.originalname).toLowerCase() || '.jpg';
  const filename = `${uuidv4()}${ext}`;

  let url;

  // Try Bunny CDN first
  if (process.env.BUNNY_API_KEY && process.env.BUNNY_STORAGE_ZONE) {
    try {
      await axios.put(
        `https://storage.bunnycdn.com/${process.env.BUNNY_STORAGE_ZONE}/${filename}`,
        file.buffer,
        {
          headers: {
            'AccessKey': process.env.BUNNY_API_KEY,
            'Content-Type': file.mimetype
          }
        }
      );
      url = `${process.env.BUNNY_CDN_URL}/${filename}`;
    } catch (error) {
      console.error('Bunny CDN error:', error.message);
      // Fall through to local storage
    }
  }

  // Fallback to local storage
  if (!url) {
    const uploadDir = path.join(__dirname, '../uploads');
    await fs.mkdir(uploadDir, { recursive: true });
    
    const filepath = path.join(uploadDir, filename);
    await fs.writeFile(filepath, file.buffer);
    
    url = `/uploads/${filename}`;
  }

  res.json({
    url,
    filename: file.originalname,
    size: file.size,
    type: file.mimetype
  });
}));

/**
 * POST /api/upload/images
 * Upload multiple images
 */
router.post('/images', upload.array('images', 10), asyncHandler(async (req, res) => {
  if (!req.files || req.files.length === 0) {
    return res.status(400).json({ error: 'No files uploaded' });
  }

  const uploads = [];
  const uploadDir = path.join(__dirname, '../uploads');
  await fs.mkdir(uploadDir, { recursive: true });

  for (const file of req.files) {
    const ext = path.extname(file.originalname).toLowerCase() || '.jpg';
    const filename = `${uuidv4()}${ext}`;
    let url;

    if (process.env.BUNNY_API_KEY && process.env.BUNNY_STORAGE_ZONE) {
      try {
        await axios.put(
          `https://storage.bunnycdn.com/${process.env.BUNNY_STORAGE_ZONE}/${filename}`,
          file.buffer,
          {
            headers: {
              'AccessKey': process.env.BUNNY_API_KEY,
              'Content-Type': file.mimetype
            }
          }
        );
        url = `${process.env.BUNNY_CDN_URL}/${filename}`;
      } catch (error) {
        // Fall through to local
      }
    }

    if (!url) {
      const filepath = path.join(uploadDir, filename);
      await fs.writeFile(filepath, file.buffer);
      url = `/uploads/${filename}`;
    }

    uploads.push({
      url,
      filename: file.originalname,
      size: file.size,
      type: file.mimetype
    });
  }

  res.json({ uploads });
}));

/**
 * DELETE /api/upload
 * Delete file
 */
router.delete('/', asyncHandler(async (req, res) => {
  const { url } = req.body;

  if (!url) {
    return res.status(400).json({ error: 'URL required' });
  }

  // Extract filename
  const filename = url.split('/').pop();

  if (process.env.BUNNY_API_KEY && process.env.BUNNY_STORAGE_ZONE && url.includes('b-cdn.net')) {
    try {
      await axios.delete(
        `https://storage.bunnycdn.com/${process.env.BUNNY_STORAGE_ZONE}/${filename}`,
        { headers: { 'AccessKey': process.env.BUNNY_API_KEY } }
      );
    } catch (error) {
      // File might not exist
    }
  } else {
    // Local file
    try {
      await fs.unlink(path.join(__dirname, '../uploads', filename));
    } catch (error) {
      // File might not exist
    }
  }

  res.json({ message: 'File deleted' });
}));

module.exports = router;
