/**
 * Settings Routes - Comprehensive Configuration
 * Tracking pixels, integrations, API credentials
 */

const express = require('express');
const router = express.Router();
const axios = require('axios');
const { pool } = require('../utils/db');
const { authenticate, requireAdmin } = require('../middleware/auth');
const { asyncHandler } = require('../middleware/errorHandler');

// All routes require authentication (except /public)
router.get('/public', asyncHandler(async (req, res) => {
  const publicKeys = [
    'site_name', 'site_logo', 'primary_color', 'secondary_color', 'custom_css',
    'ga4_measurement_id', 'meta_pixel_id', 'tiktok_pixel_id',
    'linkedin_partner_id', 'pinterest_tag_id',
    'google_ads_conversion_id', 'google_ads_conversion_label',
    'hotjar_site_id', 'clarity_project_id'
  ];

  const [settings] = await pool.execute(
    `SELECT key_name, value FROM settings WHERE key_name IN (${publicKeys.map(() => '?').join(',')})`,
    publicKeys
  );
  
  const result = {};
  settings.forEach(s => { result[s.key_name] = s.value; });

  res.json({ settings: result });
}));

// Protected routes
router.use(authenticate);

/**
 * GET /api/settings
 * Get all settings (masks sensitive values)
 */
router.get('/', asyncHandler(async (req, res) => {
  const [settings] = await pool.execute('SELECT key_name, value FROM settings');
  
  const result = {};
  const sensitiveKeys = ['api_key', 'secret', 'password'];
  
  settings.forEach(s => {
    let value = s.value;
    if (value && sensitiveKeys.some(k => s.key_name.includes(k))) {
      if (value.length > 12) {
        value = value.substring(0, 4) + '••••••••' + value.substring(value.length - 4);
      } else if (value.length > 0) {
        value = '••••••••';
      }
    }
    result[s.key_name] = value;
  });

  res.json({ settings: result });
}));

/**
 * GET /api/settings/raw
 * Get raw settings (admin only)
 */
router.get('/raw', requireAdmin, asyncHandler(async (req, res) => {
  const [settings] = await pool.execute('SELECT key_name, value FROM settings');
  const result = {};
  settings.forEach(s => { result[s.key_name] = s.value; });
  res.json({ settings: result });
}));

/**
 * PUT /api/settings
 * Update multiple settings
 */
router.put('/', requireAdmin, asyncHandler(async (req, res) => {
  const { settings } = req.body;

  if (!settings || typeof settings !== 'object') {
    return res.status(400).json({ error: 'Settings object required' });
  }

  for (const [key, value] of Object.entries(settings)) {
    // Skip masked values
    if (value && typeof value === 'string' && value.includes('••••')) continue;
    
    await pool.execute(
      'INSERT INTO settings (key_name, value) VALUES (?, ?) ON DUPLICATE KEY UPDATE value = ?',
      [key, value || null, value || null]
    );
  }

  res.json({ message: 'Settings updated' });
}));

/**
 * PUT /api/settings/tracking
 * Update all tracking pixels
 */
router.put('/tracking', requireAdmin, asyncHandler(async (req, res) => {
  const trackingKeys = [
    'ga4_measurement_id', 'meta_pixel_id', 'tiktok_pixel_id',
    'linkedin_partner_id', 'linkedin_conversion_id',
    'pinterest_tag_id', 'google_ads_conversion_id', 'google_ads_conversion_label',
    'hotjar_site_id', 'clarity_project_id'
  ];

  for (const key of trackingKeys) {
    if (req.body[key] !== undefined) {
      await pool.execute(
        'INSERT INTO settings (key_name, value) VALUES (?, ?) ON DUPLICATE KEY UPDATE value = ?',
        [key, req.body[key] || null, req.body[key] || null]
      );
    }
  }

  res.json({ message: 'Tracking settings saved' });
}));

/**
 * PUT /api/settings/mailchimp
 * Update Mailchimp with validation
 */
router.put('/mailchimp', requireAdmin, asyncHandler(async (req, res) => {
  const { api_key, list_id, tag_on_signup } = req.body;

  // Validate API key if new
  if (api_key && !api_key.includes('••••')) {
    try {
      const dc = api_key.split('-').pop();
      await axios.get(`https://${dc}.api.mailchimp.com/3.0/`, {
        headers: { 'Authorization': `Bearer ${api_key}` },
        timeout: 10000
      });
      await pool.execute(
        'INSERT INTO settings (key_name, value) VALUES (?, ?) ON DUPLICATE KEY UPDATE value = ?',
        ['mailchimp_api_key', api_key, api_key]
      );
    } catch (e) {
      return res.status(400).json({ error: 'Invalid Mailchimp API key' });
    }
  }

  if (list_id !== undefined) {
    await pool.execute(
      'INSERT INTO settings (key_name, value) VALUES (?, ?) ON DUPLICATE KEY UPDATE value = ?',
      ['mailchimp_list_id', list_id || null, list_id || null]
    );
  }

  if (tag_on_signup !== undefined) {
    await pool.execute(
      'INSERT INTO settings (key_name, value) VALUES (?, ?) ON DUPLICATE KEY UPDATE value = ?',
      ['mailchimp_tag_on_signup', tag_on_signup || null, tag_on_signup || null]
    );
  }

  res.json({ message: 'Mailchimp settings saved' });
}));

/**
 * GET /api/settings/mailchimp/lists
 */
router.get('/mailchimp/lists', requireAdmin, asyncHandler(async (req, res) => {
  const [s] = await pool.execute("SELECT value FROM settings WHERE key_name = 'mailchimp_api_key'");
  if (!s[0]?.value) return res.status(400).json({ error: 'Mailchimp not configured' });

  const apiKey = s[0].value;
  const dc = apiKey.split('-').pop();

  const r = await axios.get(`https://${dc}.api.mailchimp.com/3.0/lists?count=100`, {
    headers: { 'Authorization': `Bearer ${apiKey}` }
  });

  res.json({ lists: r.data.lists.map(l => ({ id: l.id, name: l.name, count: l.stats.member_count })) });
}));

/**
 * PUT /api/settings/kit
 * Update Kit (ConvertKit)
 */
router.put('/kit', requireAdmin, asyncHandler(async (req, res) => {
  const { api_key, form_id, tag_on_signup } = req.body;

  if (api_key && !api_key.includes('••••')) {
    try {
      await axios.get(`https://api.convertkit.com/v3/account?api_secret=${api_key}`, { timeout: 10000 });
      await pool.execute(
        'INSERT INTO settings (key_name, value) VALUES (?, ?) ON DUPLICATE KEY UPDATE value = ?',
        ['kit_api_key', api_key, api_key]
      );
    } catch (e) {
      return res.status(400).json({ error: 'Invalid Kit API key' });
    }
  }

  if (form_id !== undefined) {
    await pool.execute(
      'INSERT INTO settings (key_name, value) VALUES (?, ?) ON DUPLICATE KEY UPDATE value = ?',
      ['kit_form_id', form_id || null, form_id || null]
    );
  }

  if (tag_on_signup !== undefined) {
    await pool.execute(
      'INSERT INTO settings (key_name, value) VALUES (?, ?) ON DUPLICATE KEY UPDATE value = ?',
      ['kit_tag_on_signup', tag_on_signup || null, tag_on_signup || null]
    );
  }

  res.json({ message: 'Kit settings saved' });
}));

/**
 * GET /api/settings/kit/forms
 */
router.get('/kit/forms', requireAdmin, asyncHandler(async (req, res) => {
  const [s] = await pool.execute("SELECT value FROM settings WHERE key_name = 'kit_api_key'");
  if (!s[0]?.value) return res.status(400).json({ error: 'Kit not configured' });

  const r = await axios.get(`https://api.convertkit.com/v3/forms?api_key=${s[0].value}`);
  res.json({ forms: r.data.forms.map(f => ({ id: f.id, name: f.name })) });
}));

/**
 * GET /api/settings/kit/tags
 */
router.get('/kit/tags', requireAdmin, asyncHandler(async (req, res) => {
  const [s] = await pool.execute("SELECT value FROM settings WHERE key_name = 'kit_api_key'");
  if (!s[0]?.value) return res.status(400).json({ error: 'Kit not configured' });

  const r = await axios.get(`https://api.convertkit.com/v3/tags?api_key=${s[0].value}`);
  res.json({ tags: r.data.tags.map(t => ({ id: t.id, name: t.name })) });
}));

/**
 * PUT /api/settings/webhooks
 */
router.put('/webhooks', requireAdmin, asyncHandler(async (req, res) => {
  const webhookFields = [
    'zapier_webhook_url', 'pabbly_webhook_url',
    'zapier_webhook_new_lead', 'zapier_webhook_quiz_complete',
    'pabbly_webhook_new_lead', 'pabbly_webhook_quiz_complete'
  ];

  for (const field of webhookFields) {
    if (req.body[field] !== undefined) {
      await pool.execute(
        'INSERT INTO settings (key_name, value) VALUES (?, ?) ON DUPLICATE KEY UPDATE value = ?',
        [field, req.body[field] || null, req.body[field] || null]
      );
    }
  }

  res.json({ message: 'Webhook settings saved' });
}));

/**
 * PUT /api/settings/storage
 * Update Bunny CDN
 */
router.put('/storage', requireAdmin, asyncHandler(async (req, res) => {
  const { bunny_storage_zone, bunny_api_key, bunny_cdn_url } = req.body;

  if (bunny_storage_zone !== undefined) {
    await pool.execute(
      'INSERT INTO settings (key_name, value) VALUES (?, ?) ON DUPLICATE KEY UPDATE value = ?',
      ['bunny_storage_zone', bunny_storage_zone || null, bunny_storage_zone || null]
    );
  }

  if (bunny_api_key && !bunny_api_key.includes('••••')) {
    await pool.execute(
      'INSERT INTO settings (key_name, value) VALUES (?, ?) ON DUPLICATE KEY UPDATE value = ?',
      ['bunny_api_key', bunny_api_key, bunny_api_key]
    );
  }

  if (bunny_cdn_url !== undefined) {
    await pool.execute(
      'INSERT INTO settings (key_name, value) VALUES (?, ?) ON DUPLICATE KEY UPDATE value = ?',
      ['bunny_cdn_url', bunny_cdn_url || null, bunny_cdn_url || null]
    );
  }

  res.json({ message: 'Storage settings saved' });
}));

/**
 * PUT /api/settings/email
 * Update SendGrid
 */
router.put('/email', requireAdmin, asyncHandler(async (req, res) => {
  const { sendgrid_api_key, sendgrid_from_email } = req.body;

  if (sendgrid_api_key && !sendgrid_api_key.includes('••••')) {
    await pool.execute(
      'INSERT INTO settings (key_name, value) VALUES (?, ?) ON DUPLICATE KEY UPDATE value = ?',
      ['sendgrid_api_key', sendgrid_api_key, sendgrid_api_key]
    );
  }

  if (sendgrid_from_email !== undefined) {
    await pool.execute(
      'INSERT INTO settings (key_name, value) VALUES (?, ?) ON DUPLICATE KEY UPDATE value = ?',
      ['sendgrid_from_email', sendgrid_from_email || null, sendgrid_from_email || null]
    );
  }

  res.json({ message: 'Email settings saved' });
}));

/**
 * POST /api/settings/test-webhook
 */
router.post('/test-webhook', requireAdmin, asyncHandler(async (req, res) => {
  const { url } = req.body;
  if (!url) return res.status(400).json({ error: 'URL required' });

  try {
    await axios.post(url, {
      event: 'test',
      message: 'Test from QuizFunnel',
      timestamp: new Date().toISOString()
    }, { timeout: 10000 });
    res.json({ message: 'Test sent!' });
  } catch (e) {
    res.status(400).json({ error: `Failed: ${e.message}` });
  }
}));

module.exports = router;
