/**
 * Public Quiz Routes - Single Tenant Version
 * Routes for quiz takers (public facing)
 */

const express = require('express');
const router = express.Router();
const crypto = require('crypto');
const { v4: uuidv4 } = require('uuid');
const { pool } = require('../utils/db');
const { asyncHandler } = require('../middleware/errorHandler');
const { determineResult, calculateTags, calculateScore } = require('../utils/logic-engine');
const { isValidEmail, sanitizeEmail, sanitizeString, isValidSlug, sanitizeObject } = require('../utils/validation');

/**
 * GET /api/quiz/by-domain
 * Load quiz based on custom domain (set by middleware)
 */
router.get('/by-domain', asyncHandler(async (req, res) => {
  // Quiz was detected by domain middleware
  if (!req.quiz) {
    return res.status(404).json({ error: 'No quiz found for this domain' });
  }

  const quiz = req.quiz;

  // Increment views
  await pool.execute('UPDATE quizzes SET views = views + 1 WHERE id = ?', [quiz.id]);

  // Get questions with options
  const [questions] = await pool.execute(`
    SELECT q.id, q.uuid, q.question_text, q.question_type, q.description, q.image_url, q.required, q.order_index
    FROM questions q
    WHERE q.quiz_id = ?
    ORDER BY q.order_index
  `, [quiz.id]);

  // Get options for each question
  for (const question of questions) {
    const [options] = await pool.execute(`
      SELECT id, uuid, option_text, option_image, icon, order_index, points
      FROM question_options
      WHERE question_id = ?
      ORDER BY order_index
    `, [question.id]);
    question.options = options;
  }

  // Get pages
  const [pages] = await pool.execute(
    'SELECT id, type, slug, title, blocks, settings FROM pages WHERE quiz_id = ?',
    [quiz.id]
  );

  // Get results
  const [results] = await pool.execute(
    'SELECT id, uuid, title, slug, min_score, max_score, description, cta_text, cta_url, cta_button_text FROM results WHERE quiz_id = ? ORDER BY min_score ASC',
    [quiz.id]
  );

  // Parse JSON fields
  const parsedPages = pages.map(p => ({
    ...p,
    blocks: JSON.parse(p.blocks || '[]'),
    settings: JSON.parse(p.settings || '{}')
  }));

  res.json({
    quiz: {
      id: quiz.id,
      uuid: quiz.uuid,
      title: quiz.title,
      slug: quiz.slug,
      description: quiz.description,
      settings: JSON.parse(quiz.settings || '{}')
    },
    questions,
    pages: parsedPages,
    results
  });
}));

/**
 * GET /api/quiz/:quizSlug
 * Load quiz data for public view
 */
router.get('/:quizSlug', asyncHandler(async (req, res) => {
  const { quizSlug } = req.params;

  // Get quiz
  const [quizzes] = await pool.execute(
    'SELECT * FROM quizzes WHERE slug = ? AND published = 1',
    [quizSlug]
  );

  if (quizzes.length === 0) {
    return res.status(404).json({ error: 'Quiz not found' });
  }

  const quiz = quizzes[0];

  // Increment views
  await pool.execute('UPDATE quizzes SET views = views + 1 WHERE id = ?', [quiz.id]);

  // Get questions with options
  const [questions] = await pool.execute(`
    SELECT q.id, q.uuid, q.question_text, q.question_type, q.description, q.image_url, q.required, q.order_index
    FROM questions q
    WHERE q.quiz_id = ?
    ORDER BY q.order_index
  `, [quiz.id]);

  // Get options for each question
  for (const question of questions) {
    const [options] = await pool.execute(`
      SELECT id, uuid, option_text, option_image, icon, order_index, points
      FROM question_options
      WHERE question_id = ?
      ORDER BY order_index
    `, [question.id]);
    question.options = options;
  }

  // Get pages
  const [pages] = await pool.execute(
    'SELECT id, type, slug, title, blocks, settings FROM pages WHERE quiz_id = ?',
    [quiz.id]
  );

  // Get lead fields
  const [leadFields] = await pool.execute(
    'SELECT field_name, field_label, field_type, placeholder, options, required FROM lead_fields WHERE quiz_id = ? AND enabled = 1 ORDER BY order_index',
    [quiz.id]
  );

  // Get site settings for branding
  const [settings] = await pool.execute(
    "SELECT key_name, value FROM settings WHERE key_name IN ('site_name', 'site_logo', 'primary_color', 'secondary_color', 'custom_css')"
  );
  const branding = {};
  settings.forEach(s => branding[s.key_name] = s.value);

  res.json({
    quiz: {
      uuid: quiz.uuid,
      slug: quiz.slug,
      title: quiz.title,
      description: quiz.description,
      settings: quiz.settings ? JSON.parse(quiz.settings) : {},
      design: quiz.design ? JSON.parse(quiz.design) : {},
      leadCapturePosition: quiz.lead_capture_position,
      leadCaptureAfterQuestion: quiz.lead_capture_after_question
    },
    questions,
    landingPage: pages.find(p => p.type === 'landing'),
    leadCapturePage: pages.find(p => p.type === 'lead_capture'),
    resultPages: pages.filter(p => p.type === 'result'),
    leadFields,
    branding
  });
}));

/**
 * POST /api/quiz/:quizUuid/start
 * Start quiz session (with optional lead capture for on_landing position)
 */
router.post('/:quizUuid/start', asyncHandler(async (req, res) => {
  const { quizUuid } = req.params;
  const { name, email, phone, referrer, utm_source, utm_medium, utm_campaign, ...customFields } = req.body;

  // Get quiz
  const [quizzes] = await pool.execute('SELECT id, lead_capture_position FROM quizzes WHERE uuid = ?', [quizUuid]);
  if (quizzes.length === 0) {
    return res.status(404).json({ error: 'Quiz not found' });
  }

  const quiz = quizzes[0];
  const quizId = quiz.id;

  // Create session token
  const sessionToken = crypto.randomBytes(32).toString('hex');

  // Create session
  const [sessionResult] = await pool.execute(
    `INSERT INTO quiz_sessions (quiz_id, session_token, answers) VALUES (?, ?, '{}')`,
    [quizId, sessionToken]
  );
  const sessionId = sessionResult.insertId;

  // If email provided and lead capture is on_landing or before_quiz, create lead now
  let leadId = null;
  if (email && (quiz.lead_capture_position === 'on_landing' || quiz.lead_capture_position === 'before_quiz')) {
    const leadUuid = uuidv4();
    const ipAddress = req.ip || req.headers['x-forwarded-for'] || req.connection.remoteAddress;
    const userAgent = req.headers['user-agent'];

    const [leadResult] = await pool.execute(
      `INSERT INTO leads (quiz_id, uuid, email, name, phone, custom_fields, answers, ip_address, user_agent, referrer, utm_source, utm_medium, utm_campaign)
       VALUES (?, ?, ?, ?, ?, ?, '{}', ?, ?, ?, ?, ?, ?)`,
      [quizId, leadUuid, email, name || null, phone || null, JSON.stringify(customFields), ipAddress, userAgent, referrer || null, utm_source || null, utm_medium || null, utm_campaign || null]
    );
    leadId = leadResult.insertId;

    // Link session to lead
    await pool.execute('UPDATE quiz_sessions SET lead_id = ? WHERE id = ?', [leadId, sessionId]);
  }

  // Increment starts
  await pool.execute('UPDATE quizzes SET starts = starts + 1 WHERE id = ?', [quizId]);

  res.json({ sessionToken, leadId });
}));

/**
 * POST /api/quiz/:quizUuid/answer
 * Save answer during quiz
 */
router.post('/:quizUuid/answer', asyncHandler(async (req, res) => {
  const { quizUuid } = req.params;
  const { sessionToken, questionUuid, optionIds, textValue } = req.body;

  if (!sessionToken) {
    return res.status(400).json({ error: 'Session token required' });
  }

  // Get session
  const [sessions] = await pool.execute(
    'SELECT id, lead_id, answers FROM quiz_sessions WHERE session_token = ?',
    [sessionToken]
  );

  if (sessions.length === 0) {
    return res.status(404).json({ error: 'Session not found' });
  }

  const session = sessions[0];
  const answers = session.answers ? JSON.parse(session.answers) : {};

  // Add answer
  answers[questionUuid] = { optionIds, textValue, answeredAt: new Date().toISOString() };

  // Update session
  await pool.execute(
    'UPDATE quiz_sessions SET answers = ?, last_activity = NOW() WHERE id = ?',
    [JSON.stringify(answers), session.id]
  );

  // If lead exists, update their answers too
  if (session.lead_id) {
    await pool.execute(
      'UPDATE leads SET answers = ? WHERE id = ?',
      [JSON.stringify(answers), session.lead_id]
    );
  }

  res.json({ message: 'Answer saved' });
}));

/**
 * POST /api/quiz/:quizUuid/lead
 * Capture lead info
 */
router.post('/:quizUuid/lead', asyncHandler(async (req, res) => {
  const { quizUuid } = req.params;
  const { sessionToken, email, name, phone, ...customFields } = req.body;

  if (!sessionToken) {
    return res.status(400).json({ error: 'Session token required' });
  }

  // Validate and sanitize email
  const cleanEmail = sanitizeEmail(email);
  if (email && !isValidEmail(cleanEmail)) {
    return res.status(400).json({ error: 'Invalid email format' });
  }

  // Sanitize other inputs
  const cleanName = sanitizeString(name, 100);
  const cleanPhone = sanitizeString(phone, 20);
  const cleanCustomFields = sanitizeObject(customFields);

  // Get quiz
  const [quizzes] = await pool.execute('SELECT id FROM quizzes WHERE uuid = ?', [quizUuid]);
  if (quizzes.length === 0) {
    return res.status(404).json({ error: 'Quiz not found' });
  }

  const quizId = quizzes[0].id;

  // Get session
  const [sessions] = await pool.execute(
    'SELECT id, answers FROM quiz_sessions WHERE session_token = ? AND quiz_id = ?',
    [sessionToken, quizId]
  );

  if (sessions.length === 0) {
    return res.status(404).json({ error: 'Session not found' });
  }

  const session = sessions[0];
  const answers = session.answers ? JSON.parse(session.answers) : {};

  // Get tracking info
  const ipAddress = req.ip || req.headers['x-forwarded-for'] || req.connection.remoteAddress;
  const userAgent = sanitizeString(req.headers['user-agent'], 500);
  const referrer = sanitizeString(req.headers.referer, 500);

  // Create lead
  const leadUuid = uuidv4();
  const [result] = await pool.execute(
    `INSERT INTO leads (quiz_id, uuid, email, name, phone, custom_fields, answers, ip_address, user_agent, referrer, utm_source, utm_medium, utm_campaign)
     VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
    [quizId, leadUuid, cleanEmail || null, cleanName || null, cleanPhone || null, JSON.stringify(cleanCustomFields), JSON.stringify(answers), ipAddress, userAgent, referrer, sanitizeString(req.body.utm_source, 100) || null, sanitizeString(req.body.utm_medium, 100) || null, sanitizeString(req.body.utm_campaign, 100) || null]
  );

  // Link session to lead
  await pool.execute(
    'UPDATE quiz_sessions SET lead_id = ? WHERE id = ?',
    [result.insertId, session.id]
  );

  res.json({
    leadId: leadUuid,
    message: 'Lead captured'
  });
}));

/**
 * POST /api/quiz/:quizUuid/complete
 * Complete quiz and get results
 */
router.post('/:quizUuid/complete', asyncHandler(async (req, res) => {
  const { quizUuid } = req.params;
  const { sessionToken, finalScore } = req.body;

  if (!sessionToken) {
    return res.status(400).json({ error: 'Session token required' });
  }

  // Get quiz
  const [quizzes] = await pool.execute('SELECT id, scoring_type FROM quizzes WHERE uuid = ?', [quizUuid]);
  if (quizzes.length === 0) {
    return res.status(404).json({ error: 'Quiz not found' });
  }

  const quiz = quizzes[0];
  const quizId = quiz.id;

  // Get session
  const [sessions] = await pool.execute(
    'SELECT id, lead_id, answers FROM quiz_sessions WHERE session_token = ? AND quiz_id = ?',
    [sessionToken, quizId]
  );

  if (sessions.length === 0) {
    return res.status(404).json({ error: 'Session not found' });
  }

  const session = sessions[0];
  const answers = session.answers ? JSON.parse(session.answers) : {};

  // Get all options with points for score calculation
  const [allOptions] = await pool.execute(`
    SELECT qo.id, qo.uuid, qo.points, q.uuid as question_uuid
    FROM question_options qo
    JOIN questions q ON qo.question_id = q.id
    WHERE q.quiz_id = ?
  `, [quizId]);

  // Calculate score from answers
  let score = finalScore || 0;
  
  if (!finalScore) {
    // Calculate from answers if not provided
    for (const [questionUuid, answer] of Object.entries(answers)) {
      if (answer.optionIds && Array.isArray(answer.optionIds)) {
        for (const optionId of answer.optionIds) {
          const option = allOptions.find(o => o.id === optionId);
          if (option && option.points) {
            score += option.points;
          }
        }
      }
    }
  }

  // Get logic rules
  const [rules] = await pool.execute(
    'SELECT * FROM logic_rules WHERE quiz_id = ? AND enabled = 1 ORDER BY priority DESC',
    [quizId]
  );

  // Get result pages with settings
  const [resultPages] = await pool.execute(
    'SELECT id, slug, title, blocks, is_default, settings, mailchimp_tag, kit_tag FROM pages WHERE quiz_id = ? AND type = "result"',
    [quizId]
  );

  // Calculate tags
  const selectedOptions = [];
  for (const [questionUuid, answer] of Object.entries(answers)) {
    if (answer.optionIds) {
      for (const optionId of answer.optionIds) {
        const option = allOptions.find(o => o.id === optionId);
        if (option) selectedOptions.push(option);
      }
    }
  }
  const tags = calculateTags(selectedOptions, rules, answers);

  // Determine result page based on score
  const resultPage = determineResult(answers, rules, resultPages, score);

  // Update session
  await pool.execute(
    'UPDATE quiz_sessions SET completed = 1 WHERE id = ?',
    [session.id]
  );

  // Update lead if exists
  if (session.lead_id) {
    await pool.execute(
      'UPDATE leads SET completed_quiz = 1, result_page_id = ?, tags = ?, score = ?, answers = ? WHERE id = ?',
      [resultPage.id, JSON.stringify(tags), score, JSON.stringify(answers), session.lead_id]
    );

    // Trigger integrations asynchronously
    triggerIntegrations(session.lead_id).catch(err => console.error('Integration error:', err));
  }

  // Increment completions
  await pool.execute('UPDATE quizzes SET completions = completions + 1 WHERE id = ?', [quizId]);

  res.json({
    resultPage: {
      id: resultPage.id,
      slug: resultPage.slug,
      title: resultPage.title,
      blocks: resultPage.blocks
    },
    score,
    tags
  });
}));

/**
 * Trigger integrations for lead
 */
async function triggerIntegrations(leadId) {
  try {
    const { syncLeadToAllIntegrations } = require('../services/sync');
    await syncLeadToAllIntegrations(leadId);
  } catch (error) {
    console.error('Integration sync error:', error);
  }
}

/**
 * GET /api/quiz/:quizUuid/result/:resultSlug
 * Get specific result page (for direct links)
 */
router.get('/:quizUuid/result/:resultSlug', asyncHandler(async (req, res) => {
  const { quizUuid, resultSlug } = req.params;

  const [pages] = await pool.execute(`
    SELECT p.* FROM pages p
    JOIN quizzes q ON p.quiz_id = q.id
    WHERE q.uuid = ? AND p.slug = ? AND p.type = 'result'
  `, [quizUuid, resultSlug]);

  if (pages.length === 0) {
    return res.status(404).json({ error: 'Result page not found' });
  }

  res.json({ resultPage: pages[0] });
}));

module.exports = router;
