/**
 * Integrations Routes - Single Tenant Version
 * Mailchimp, Kit, Google Sheets, Webhooks
 */

const express = require('express');
const router = express.Router();
const axios = require('axios');
const { pool } = require('../utils/db');
const { encrypt, decrypt, encryptObject, decryptObject } = require('../utils/encryption');
const { authenticate, requireAdmin } = require('../middleware/auth');
const { asyncHandler } = require('../middleware/errorHandler');

// All routes require authentication
router.use(authenticate);

/**
 * GET /api/integrations
 * List all integrations
 */
router.get('/', asyncHandler(async (req, res) => {
  const [integrations] = await pool.execute(
    'SELECT id, platform, name, enabled, last_sync, last_error, created_at FROM integrations'
  );

  res.json({ integrations });
}));

/**
 * POST /api/integrations/mailchimp
 * Connect Mailchimp
 */
router.post('/mailchimp', requireAdmin, asyncHandler(async (req, res) => {
  const { apiKey, name } = req.body;

  if (!apiKey) {
    return res.status(400).json({ error: 'API key required' });
  }

  // Verify API key
  const datacenter = apiKey.split('-').pop();
  try {
    const response = await axios.get(`https://${datacenter}.api.mailchimp.com/3.0/`, {
      headers: { 'Authorization': `Bearer ${apiKey}` }
    });

    // Get lists
    const listsResponse = await axios.get(`https://${datacenter}.api.mailchimp.com/3.0/lists?count=100`, {
      headers: { 'Authorization': `Bearer ${apiKey}` }
    });

    const encryptedKey = encrypt(apiKey);

    // Upsert integration
    await pool.execute(`
      INSERT INTO integrations (platform, name, credentials, settings, enabled)
      VALUES ('mailchimp', ?, ?, ?, 1)
      ON DUPLICATE KEY UPDATE name = ?, credentials = ?, settings = ?, enabled = 1, last_error = NULL
    `, [
      name || 'Mailchimp',
      encryptedKey,
      JSON.stringify({ datacenter, accountName: response.data.account_name }),
      name || 'Mailchimp',
      encryptedKey,
      JSON.stringify({ datacenter, accountName: response.data.account_name })
    ]);

    res.json({
      message: 'Mailchimp connected',
      lists: listsResponse.data.lists.map(l => ({ id: l.id, name: l.name, memberCount: l.stats.member_count }))
    });
  } catch (error) {
    res.status(400).json({ error: 'Invalid API key or connection failed' });
  }
}));

/**
 * GET /api/integrations/mailchimp/lists
 * Get Mailchimp lists
 */
router.get('/mailchimp/lists', asyncHandler(async (req, res) => {
  const [integrations] = await pool.execute(
    'SELECT credentials, settings FROM integrations WHERE platform = "mailchimp" AND enabled = 1'
  );

  if (integrations.length === 0) {
    return res.status(404).json({ error: 'Mailchimp not connected' });
  }

  const apiKey = decrypt(integrations[0].credentials);
  const settings = JSON.parse(integrations[0].settings || '{}');

  const response = await axios.get(`https://${settings.datacenter}.api.mailchimp.com/3.0/lists?count=100`, {
    headers: { 'Authorization': `Bearer ${apiKey}` }
  });

  res.json({
    lists: response.data.lists.map(l => ({ id: l.id, name: l.name, memberCount: l.stats.member_count }))
  });
}));

/**
 * POST /api/integrations/kit
 * Connect Kit (ConvertKit)
 */
router.post('/kit', requireAdmin, asyncHandler(async (req, res) => {
  const { apiKey, apiSecret, name } = req.body;

  if (!apiKey) {
    return res.status(400).json({ error: 'API key required' });
  }

  try {
    // Verify connection
    const response = await axios.get(`https://api.convertkit.com/v3/account?api_secret=${apiSecret || apiKey}`);

    // Get forms
    const formsResponse = await axios.get(`https://api.convertkit.com/v3/forms?api_key=${apiKey}`);

    // Get tags
    const tagsResponse = await axios.get(`https://api.convertkit.com/v3/tags?api_key=${apiKey}`);

    const encryptedCreds = encryptObject({ apiKey, apiSecret: apiSecret || apiKey });

    await pool.execute(`
      INSERT INTO integrations (platform, name, credentials, settings, enabled)
      VALUES ('kit', ?, ?, ?, 1)
      ON DUPLICATE KEY UPDATE name = ?, credentials = ?, settings = ?, enabled = 1, last_error = NULL
    `, [
      name || 'Kit',
      encryptedCreds,
      JSON.stringify({ accountName: response.data.name }),
      name || 'Kit',
      encryptedCreds,
      JSON.stringify({ accountName: response.data.name })
    ]);

    res.json({
      message: 'Kit connected',
      forms: formsResponse.data.forms.map(f => ({ id: f.id, name: f.name })),
      tags: tagsResponse.data.tags.map(t => ({ id: t.id, name: t.name }))
    });
  } catch (error) {
    res.status(400).json({ error: 'Invalid credentials or connection failed' });
  }
}));

/**
 * POST /api/integrations/sheets
 * Connect Google Sheets
 */
router.post('/sheets', requireAdmin, asyncHandler(async (req, res) => {
  const { spreadsheetId, name } = req.body;

  if (!spreadsheetId) {
    return res.status(400).json({ error: 'Spreadsheet ID required' });
  }

  // Check if Google Sheets is configured in env
  if (!process.env.GOOGLE_SERVICE_ACCOUNT_EMAIL || !process.env.GOOGLE_PRIVATE_KEY) {
    return res.status(400).json({ error: 'Google Sheets not configured on server' });
  }

  try {
    const { google } = require('googleapis');
    const auth = new google.auth.GoogleAuth({
      credentials: {
        client_email: process.env.GOOGLE_SERVICE_ACCOUNT_EMAIL,
        private_key: process.env.GOOGLE_PRIVATE_KEY.replace(/\\n/g, '\n')
      },
      scopes: ['https://www.googleapis.com/auth/spreadsheets']
    });

    const sheets = google.sheets({ version: 'v4', auth });
    
    // Verify access
    const response = await sheets.spreadsheets.get({
      spreadsheetId,
      fields: 'properties.title,sheets.properties.title'
    });

    const settings = {
      spreadsheetId,
      spreadsheetName: response.data.properties.title,
      sheets: response.data.sheets.map(s => s.properties.title)
    };

    await pool.execute(`
      INSERT INTO integrations (platform, name, credentials, settings, enabled)
      VALUES ('sheets', ?, ?, ?, 1)
      ON DUPLICATE KEY UPDATE name = ?, credentials = ?, settings = ?, enabled = 1, last_error = NULL
    `, [
      name || 'Google Sheets',
      '', // No credentials needed - using service account
      JSON.stringify(settings),
      name || 'Google Sheets',
      '',
      JSON.stringify(settings)
    ]);

    res.json({
      message: 'Google Sheets connected',
      spreadsheetName: response.data.properties.title,
      sheets: response.data.sheets.map(s => s.properties.title)
    });
  } catch (error) {
    console.error('Sheets error:', error);
    res.status(400).json({ error: 'Cannot access spreadsheet. Make sure to share it with the service account email.' });
  }
}));

/**
 * POST /api/integrations/webhook
 * Add webhook
 */
router.post('/webhook', requireAdmin, asyncHandler(async (req, res) => {
  const { url, name, headers } = req.body;

  if (!url) {
    return res.status(400).json({ error: 'Webhook URL required' });
  }

  // Test webhook
  try {
    await axios.post(url, { event: 'test', timestamp: new Date().toISOString() }, {
      headers: { 'Content-Type': 'application/json', ...headers },
      timeout: 10000
    });
  } catch (error) {
    // Don't fail if webhook test fails, just log it
    console.log('Webhook test failed:', error.message);
  }

  const settings = { url, headers: headers || {} };

  await pool.execute(`
    INSERT INTO integrations (platform, name, credentials, settings, enabled)
    VALUES ('webhook', ?, ?, ?, 1)
    ON DUPLICATE KEY UPDATE name = ?, credentials = ?, settings = ?, enabled = 1, last_error = NULL
  `, [name || 'Webhook', '', JSON.stringify(settings), name || 'Webhook', '', JSON.stringify(settings)]);

  res.json({ message: 'Webhook added' });
}));

/**
 * PUT /api/integrations/:integrationId
 * Update integration
 */
router.put('/:integrationId', requireAdmin, asyncHandler(async (req, res) => {
  const { integrationId } = req.params;
  const { name, enabled, settings } = req.body;

  const updates = [];
  const values = [];

  if (name !== undefined) { updates.push('name = ?'); values.push(name); }
  if (enabled !== undefined) { updates.push('enabled = ?'); values.push(enabled ? 1 : 0); }
  if (settings !== undefined) { updates.push('settings = ?'); values.push(JSON.stringify(settings)); }

  if (updates.length === 0) {
    return res.status(400).json({ error: 'No fields to update' });
  }

  values.push(integrationId);
  await pool.execute(`UPDATE integrations SET ${updates.join(', ')} WHERE id = ?`, values);

  res.json({ message: 'Integration updated' });
}));

/**
 * DELETE /api/integrations/:integrationId
 * Delete integration
 */
router.delete('/:integrationId', requireAdmin, asyncHandler(async (req, res) => {
  const { integrationId } = req.params;
  await pool.execute('DELETE FROM integrations WHERE id = ?', [integrationId]);
  res.json({ message: 'Integration deleted' });
}));

/**
 * POST /api/integrations/:integrationId/quiz/:quizId
 * Link integration to quiz
 */
router.post('/:integrationId/quiz/:quizId', asyncHandler(async (req, res) => {
  const { integrationId, quizId } = req.params;
  const { listId, formId, sheetName, tagMapping, fieldMapping } = req.body;

  await pool.execute(`
    INSERT INTO quiz_integrations (quiz_id, integration_id, list_id, form_id, sheet_name, tag_mapping, field_mapping, enabled)
    VALUES (?, ?, ?, ?, ?, ?, ?, 1)
    ON DUPLICATE KEY UPDATE list_id = ?, form_id = ?, sheet_name = ?, tag_mapping = ?, field_mapping = ?, enabled = 1
  `, [
    quizId, integrationId, listId || null, formId || null, sheetName || null,
    tagMapping ? JSON.stringify(tagMapping) : null, fieldMapping ? JSON.stringify(fieldMapping) : null,
    listId || null, formId || null, sheetName || null,
    tagMapping ? JSON.stringify(tagMapping) : null, fieldMapping ? JSON.stringify(fieldMapping) : null
  ]);

  res.json({ message: 'Integration linked to quiz' });
}));

/**
 * DELETE /api/integrations/:integrationId/quiz/:quizId
 * Unlink integration from quiz
 */
router.delete('/:integrationId/quiz/:quizId', asyncHandler(async (req, res) => {
  const { integrationId, quizId } = req.params;
  await pool.execute('DELETE FROM quiz_integrations WHERE integration_id = ? AND quiz_id = ?', [integrationId, quizId]);
  res.json({ message: 'Integration unlinked from quiz' });
}));

module.exports = router;
