/**
 * Auth Middleware - Single Tenant Version
 * Simple JWT authentication, no multi-tenant complexity
 */

const jwt = require('jsonwebtoken');
const { pool } = require('../utils/db');

// SECURITY: No default fallback - must be set in .env
const JWT_SECRET = process.env.JWT_SECRET;
const JWT_REFRESH_SECRET = process.env.JWT_REFRESH_SECRET;

// Fail fast if secrets not configured
if (!JWT_SECRET || JWT_SECRET.length < 32) {
  console.error('FATAL: JWT_SECRET must be set in .env (minimum 32 characters)');
  process.exit(1);
}

if (!JWT_REFRESH_SECRET || JWT_REFRESH_SECRET.length < 32) {
  console.error('FATAL: JWT_REFRESH_SECRET must be set in .env (minimum 32 characters)');
  process.exit(1);
}

/**
 * Generate access and refresh tokens
 */
function generateTokens(payload) {
  const accessToken = jwt.sign(payload, JWT_SECRET, { expiresIn: '24h' });
  const refreshToken = jwt.sign(payload, JWT_REFRESH_SECRET, { expiresIn: '7d' });
  return { accessToken, refreshToken };
}

/**
 * Verify JWT token
 */
function verifyToken(token, secret = JWT_SECRET) {
  try {
    return jwt.verify(token, secret);
  } catch (error) {
    return null;
  }
}

/**
 * Authenticate user middleware
 */
async function authenticate(req, res, next) {
  const authHeader = req.headers.authorization;
  
  if (!authHeader || !authHeader.startsWith('Bearer ')) {
    return res.status(401).json({ error: 'No token provided' });
  }

  const token = authHeader.substring(7);
  const decoded = verifyToken(token);

  if (!decoded) {
    return res.status(401).json({ error: 'Invalid or expired token' });
  }

  // Get user from database
  try {
    const [users] = await pool.execute(
      'SELECT id, email, name, role FROM users WHERE id = ?',
      [decoded.userId]
    );

    if (users.length === 0) {
      return res.status(401).json({ error: 'User not found' });
    }

    req.user = users[0];
    req.userId = users[0].id;
    next();
  } catch (error) {
    console.error('Auth error:', error);
    res.status(500).json({ error: 'Authentication failed' });
  }
}

/**
 * Require admin role
 */
function requireAdmin(req, res, next) {
  if (req.user.role !== 'admin') {
    return res.status(403).json({ error: 'Admin access required' });
  }
  next();
}

/**
 * Require editor or admin role
 */
function requireEditor(req, res, next) {
  if (req.user.role === 'viewer') {
    return res.status(403).json({ error: 'Editor access required' });
  }
  next();
}

/**
 * Optional authentication - doesn't fail if no token
 */
async function optionalAuth(req, res, next) {
  const authHeader = req.headers.authorization;
  
  if (authHeader && authHeader.startsWith('Bearer ')) {
    const token = authHeader.substring(7);
    const decoded = verifyToken(token);

    if (decoded) {
      try {
        const [users] = await pool.execute(
          'SELECT id, email, name, role FROM users WHERE id = ?',
          [decoded.userId]
        );
        if (users.length > 0) {
          req.user = users[0];
          req.userId = users[0].id;
        }
      } catch (error) {
        // Ignore errors for optional auth
      }
    }
  }
  
  next();
}

module.exports = {
  generateTokens,
  verifyToken,
  authenticate,
  requireAdmin,
  requireEditor,
  optionalAuth
};
