#!/usr/bin/env node
/**
 * QuizFunnel Deployment Helper
 * Checks configuration and helps troubleshoot
 */

require('dotenv').config();
const fs = require('fs');

console.log('\n🔍 QuizFunnel Deployment Check\n');
console.log('=' .repeat(50));

// Check .env exists
if (!fs.existsSync('.env')) {
  console.log('❌ .env file not found');
  console.log('   Copy .env.example to .env and fill in values');
  process.exit(1);
}
console.log('✅ .env file exists');

// Check required variables
const required = ['DB_HOST', 'DB_USER', 'DB_PASSWORD', 'DB_NAME', 'JWT_SECRET'];
const missing = required.filter(v => !process.env[v]);

if (missing.length > 0) {
  console.log(`❌ Missing required variables: ${missing.join(', ')}`);
  process.exit(1);
}
console.log('✅ Required environment variables set');

// Check database connection
async function checkDatabase() {
  try {
    const mysql = require('mysql2/promise');
    const connection = await mysql.createConnection({
      host: process.env.DB_HOST,
      user: process.env.DB_USER,
      password: process.env.DB_PASSWORD,
      database: process.env.DB_NAME
    });
    
    // Check if tables exist
    const [tables] = await connection.execute('SHOW TABLES');
    await connection.end();
    
    if (tables.length === 0) {
      console.log('⚠️  Database connected but no tables found');
      console.log('   Import database.sql via phpMyAdmin');
      return false;
    }
    
    console.log(`✅ Database connected (${tables.length} tables)`);
    return true;
  } catch (error) {
    console.log('❌ Database connection failed:', error.message);
    return false;
  }
}

// Check admin exists
async function checkAdmin() {
  try {
    const { pool } = require('./utils/db');
    const [users] = await pool.execute('SELECT COUNT(*) as count FROM users WHERE role = "admin"');
    
    if (users[0].count === 0) {
      console.log('⚠️  No admin user found');
      console.log('   Run: node setup.js');
      return false;
    }
    
    console.log(`✅ Admin user exists`);
    return true;
  } catch (error) {
    return false;
  }
}

// Check uploads folder
function checkUploads() {
  const uploadsPath = './uploads';
  if (!fs.existsSync(uploadsPath)) {
    fs.mkdirSync(uploadsPath, { recursive: true });
  }
  
  try {
    fs.accessSync(uploadsPath, fs.constants.W_OK);
    console.log('✅ Uploads folder writable');
    return true;
  } catch {
    console.log('⚠️  Uploads folder not writable');
    console.log('   Run: chmod 755 uploads');
    return false;
  }
}

// Run checks
(async () => {
  checkUploads();
  const dbOk = await checkDatabase();
  if (dbOk) await checkAdmin();
  
  console.log('\n' + '=' .repeat(50));
  console.log('App URL:', process.env.APP_URL || `http://localhost:${process.env.PORT || 3000}`);
  console.log('Admin:  ', (process.env.APP_URL || `http://localhost:${process.env.PORT || 3000}`) + '/admin');
  console.log('');
  
  process.exit(0);
})();
